<?php

namespace Database\Seeders\Demo;

use App\Models\Feature;
use App\Models\Package;
use App\Models\Customer;
use App\Models\UserPackage;
use App\Models\PackageFeature;
use App\Services\HelperService;
use Illuminate\Database\Seeder;
use App\Models\UserPackageLimit;
use Illuminate\Support\Facades\Log;

class PackageDataSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        $this->createPackages();
    }

    /** Add Packages Data */
    public function createPackages() {
        /** Packages Data */
        $packageLastId = Package::latest()->pluck('id')->first();
        if(empty($packageLastId)){
            $packageLastId = 0;
        }

        $this->createPackage($packageLastId);

        // Get Last 3 Packages ID
        $packagesId = Package::latest()->limit(3)->pluck('id')->toArray();

        /** Features Data */
        $this->createFeatures();

        /** Package Features Data */
        $this->createPackageFeatures($packagesId);

        /** User Packages Data */
        $this->createUserPackages($packagesId);

    }

    /** Packages Demo Functions */
    // Create Packages
    private function createPackage($packageLastId) {
        $packageData = array(
            [
                'id'            => $packageLastId + 1,
                'name'          => 'Free Listing and Feature',
                'package_type'  => 'free',
                'duration'      => 240,
                'status'        => 1,
                'created_at'    => now(),
                'updated_at'    => now(),
            ],
            [
                'id'            => $packageLastId + 2,
                'name'          => 'Only Premium Features',
                'package_type'  => 'free',
                'duration'      => 240,
                'status'        => 1,
                'created_at'    => now(),
                'updated_at'    => now(),
            ],
            [
                'id'            => $packageLastId + 3,
                'name'          => 'All Features Unlimited',
                'package_type'  => 'free',
                'duration'      => 120,
                'status'        => 1,
                'created_at'    => now(),
                'updated_at'    => now(),
            ]
        );
        Package::upsert($packageData,['id'],['name','package_type','duration','status']);
    }

    // Create Features
    private function createFeatures() {
        // Check All Features Exists
        $featureNames = HelperService::getFeatureNames();
        $featuresQuery = Feature::whereIn('name',$featureNames);
        $featuresCount = $featuresQuery->count();
        $features = $featuresQuery->get();

        if(empty($features) || $featuresCount != 7){
            Log::error('No features found');
            /** Add Data */
            $featureData = array(
                ['id' => 1, 'name' => $featureNames[0], 'status' => 1],
                ['id' => 2, 'name' => $featureNames[1], 'status' => 1],
                ['id' => 3, 'name' => $featureNames[2], 'status' => 1],
                ['id' => 4, 'name' => $featureNames[3], 'status' => 1],
                ['id' => 5, 'name' => $featureNames[4], 'status' => 1],
                ['id' => 6, 'name' => $featureNames[5], 'status' => 1],
                ['id' => 7, 'name' => $featureNames[6], 'status' => 1],
            );
            Feature::upsert($featureData,['id'],['name','status']);
            $features = Feature::get();
        }
    }

    /** Add Package Features Data */
    private function createPackageFeatures($packagesId) {

        $packageFeaturesData = array();
        foreach($packagesId as $key => $packageId){
            if($key == 0){
                $packageFeaturesData[] = array(
                    [
                        'package_id' => $packageId,
                        'feature_id' => HelperService::getFeatureId('property_list'),
                        'limit_type' => 'limited',
                        'limit' => 5
                    ],
                    [
                        'package_id' => $packageId,
                        'feature_id' => HelperService::getFeatureId('project_list'),
                        'limit_type' => 'limited',
                        'limit' => 5
                    ],
                    [
                        'package_id' => $packageId,
                        'feature_id' => HelperService::getFeatureId('property_feature'),
                        'limit_type' => 'limited',
                        'limit' => 5
                    ],
                    [
                        'package_id' => $packageId,
                        'feature_id' => HelperService::getFeatureId('project_feature'),
                        'limit_type' => 'limited',
                        'limit' => 5
                    ]
                );
            } else if ($key == 1) {
                $packageFeaturesData[] = array(
                    [
                        'package_id' => $packageId,
                        'feature_id' => HelperService::getFeatureId('mortgage_calculator_detail'),
                        'limit_type' => 'unlimited'
                    ],
                    [
                        'package_id' => $packageId,
                        'feature_id' => HelperService::getFeatureId('premium_properties'),
                        'limit_type' => 'unlimited'
                    ],
                    [
                        'package_id' => $packageId,
                        'feature_id' => HelperService::getFeatureId('project_access'),
                        'limit_type' => 'unlimited'
                    ]
                );
            }else {
                $packageFeaturesData[] = array(
                    [
                        'package_id' => $packageId,
                        'feature_id' => HelperService::getFeatureId('property_list'),
                        'limit_type' => 'unlimited'
                    ],
                    [
                        'package_id' => $packageId,
                        'feature_id' => HelperService::getFeatureId('project_list'),
                        'limit_type' => 'unlimited'
                    ],
                    [
                        'package_id' => $packageId,
                        'feature_id' => HelperService::getFeatureId('property_feature'),
                        'limit_type' => 'unlimited'
                    ],
                    [
                        'package_id' => $packageId,
                        'feature_id' => HelperService::getFeatureId('project_feature'),
                        'limit_type' => 'unlimited'
                    ],
                    [
                        'package_id' => $packageId,
                        'feature_id' => HelperService::getFeatureId('mortgage_calculator_detail'),
                        'limit_type' => 'unlimited'
                    ],
                    [
                        'package_id' => $packageId,
                        'feature_id' => HelperService::getFeatureId('premium_properties'),
                        'limit_type' => 'unlimited'
                    ],
                    [
                        'package_id' => $packageId,
                        'feature_id' => HelperService::getFeatureId('project_access'),
                        'limit_type' => 'unlimited'
                    ]
                );
            }
        }
        PackageFeature::upsert($packageFeaturesData,['package_id','feature_id'],['limit_type','limit']);
    }

    /** Add User Packages Data */
    private function createUserPackages($packagesId) {
        // Get Last 3 Users ID
        $lastUsersID = Customer::latest()->limit(3)->pluck('id')->toArray();

        // Create User Package Data
        $userPackagesData = array();
        foreach($lastUsersID as $key => $userId){
            $selectedPackageId = $packagesId[rand(0,2)]; // Get Random Package ID
            $getPackage = Package::find($selectedPackageId); // Get Package Data

            // Create Or Update User Package Data
            $userPackagesData = array(
                'user_id' => $userId,
                'package_id' => $selectedPackageId,
                'start_date' => now(),
                'end_date' => now()->addDays($getPackage->duration),
                'created_at' => now(),
                'updated_at' => now(),
            );
            $userPackage = UserPackage::updateOrCreate(['user_id' => $userId,'package_id' => $selectedPackageId],$userPackagesData);

            // Get Limited Count Feature
            $packageFeatures = PackageFeature::where(['package_id' => $selectedPackageId, 'limit_type' => 'limited'])->get();

            // If Limited Count Feature Found Then Create User Package Limit Data
            if(collect($packageFeatures)->isNotEmpty()){
                $userPackageLimitData = array();
                foreach ($packageFeatures as $key => $feature) {
                    $userPackageLimitData[] = array(
                        'user_package_id' => $userPackage->id,
                        'package_feature_id' => $feature->id,
                        'total_limit' => $feature->limit,
                        'used_limit' => 0,
                        'created_at' => now(),
                        'updated_at' => now()
                    );
                }

                if(!empty($userPackageLimitData)){
                    UserPackageLimit::insert($userPackageLimitData);
                }
            }
        }
    }
}
